/*******************************************************************************
 * Copyright (c) 2011, 2025 Google, Inc. and others.
 *
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License 2.0 which is available at
 * https://www.eclipse.org/legal/epl-2.0.
 *
 * SPDX-License-Identifier: EPL-2.0
 *
 * Contributors:
 *    Google, Inc. - initial API and implementation
 *******************************************************************************/
package org.eclipse.wb.internal.swing.model.property.editor.models.spinner;

import org.eclipse.wb.internal.core.DesignerPlugin;
import org.eclipse.wb.internal.core.utils.dialogfields.AbstractValidationTitleAreaDialog;
import org.eclipse.wb.internal.core.utils.ui.GridDataFactory;
import org.eclipse.wb.internal.core.utils.ui.GridLayoutFactory;
import org.eclipse.wb.internal.swing.Activator;
import org.eclipse.wb.internal.swing.model.ModelMessages;
import org.eclipse.wb.internal.swing.utils.SwingUtils;

import org.eclipse.swt.SWT;
import org.eclipse.swt.custom.CTabFolder;
import org.eclipse.swt.custom.CTabItem;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Group;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Shell;

import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.CompletableFuture;

/**
 * The dialog for editing {@link javax.swing.SpinnerModel SpinnerModel}.
 *
 * @author scheglov_ke
 * @coverage swing.property.editor
 */
public final class SpinnerModelDialog extends AbstractValidationTitleAreaDialog {
	private final SpinnerModelValue m_modelValue;
	private SpinnerModelCanvas m_spinnerPreview;
	private String m_source;

	////////////////////////////////////////////////////////////////////////////
	//
	// Constructor
	//
	////////////////////////////////////////////////////////////////////////////
	public SpinnerModelDialog(Shell parentShell, String shellText, SpinnerModelValue modelValue) {
		super(parentShell,
				Activator.getDefault(),
				shellText,
				ModelMessages.SpinnerModelDialog_title,
				null,
				ModelMessages.SpinnerModelDialog_message);
		m_modelValue = modelValue;
	}

	////////////////////////////////////////////////////////////////////////////
	//
	// Access
	//
	////////////////////////////////////////////////////////////////////////////
	/**
	 * @return the source of {@link javax.swing.SpinnerModel SpinnerModel} that
	 *         represents user selections.
	 */
	public String getSource() {
		return m_source;
	}

	////////////////////////////////////////////////////////////////////////////
	//
	// GUI
	//
	////////////////////////////////////////////////////////////////////////////
	private final List<AbstractSpinnerComposite> m_composites = new ArrayList<>();
	private CTabFolder m_tabFolder;

	//private Combo m_typeCombo;
	@Override
	protected void createControls(Composite container) {
		GridLayoutFactory.create(container);
		m_tabFolder = new CTabFolder(container, SWT.NONE);
		m_tabFolder.addSelectionListener(new SelectionAdapter() {
			@Override
			public void widgetSelected(SelectionEvent e) {
				validateAll();
			}
		});
		GridDataFactory.create(m_tabFolder).grab().fill();
		// add composites
		{
			m_composites.add(new ListSpinnerComposite(m_tabFolder, this));
			m_composites.add(new NumberSpinnerComposite(m_tabFolder, this));
			m_composites.add(new DateSpinnerComposite(m_tabFolder, this));
		}
		// create tab for each spinner composite
		for (AbstractSpinnerComposite composite : m_composites) {
			// create tab
			CTabItem tabItem = new CTabItem(m_tabFolder, SWT.NONE);
			tabItem.setControl(composite);
			tabItem.setText(composite.getTitle());
			// select tab
			SwingUtils.runLogLater(() -> {
				CompletableFuture<Void> result = composite.setModelValue(m_modelValue);
				if (result != null) {
					result.thenAccept(ignore -> m_tabFolder.setSelection(tabItem));
				}
			});
		}
		// preview
		createPreviewComposite(container);
	}

	/**
	 * Creates {@link Composite} with {@link javax.swing.JSpinner JSpinner} for
	 * preview.
	 */
	private void createPreviewComposite(Composite parent) {
		Group previewGroup = new Group(parent, SWT.NONE);
		GridDataFactory.create(previewGroup).fill();
		GridLayoutFactory.create(previewGroup).columns(2);
		previewGroup.setText(ModelMessages.SpinnerModelDialog_preview);
		// hint
		{
			Label label = new Label(previewGroup, SWT.NONE);
			GridDataFactory.create(label).spanH(2);
			label.setText(ModelMessages.SpinnerModelDialog_hint);
		}
		{
			Label label = new Label(previewGroup, SWT.NONE);
			configureColumn_1(label);
			label.setText(ModelMessages.SpinnerModelDialog_test);
		}
		{
			// two clicks needed to focus AWT component, see:
			// http://bugs.sun.com/bugdatabase/view_bug.do?bug_id=6523306
			m_spinnerPreview = new SpinnerModelCanvas(previewGroup, SWT.NONE);
			m_spinnerPreview.populate();
			GridDataFactory.create(m_spinnerPreview).grab().fill();
		}
	}

	@Override
	protected void okPressed() {
		try {
			AbstractSpinnerComposite spinnerComposite = getSelectedSpinnerComposite();
			m_source = spinnerComposite.getSource();
		} catch (Throwable e) {
			DesignerPlugin.log(e);
			return;
		}
		// close dialog
		super.okPressed();
	}

	////////////////////////////////////////////////////////////////////////////
	//
	// Utils
	//
	////////////////////////////////////////////////////////////////////////////
	@Override
	protected String validate() throws Exception {
		AbstractSpinnerComposite spinnerComposite = getSelectedSpinnerComposite();
		// validate
		{
			String message = spinnerComposite.validate();
			if (message != null) {
				m_spinnerPreview.disable();
				return message;
			}
		}
		// configure spinner
		{
			m_spinnerPreview.setSpinnerModel(spinnerComposite.getModelValue());
		}
		// OK
		return null;
	}

	/**
	 * @return the selected {@link AbstractSpinnerComposite}.
	 */
	private AbstractSpinnerComposite getSelectedSpinnerComposite() {
		int index = m_tabFolder.getSelectionIndex();
		AbstractSpinnerComposite spinnerComposite = m_composites.get(index);
		return spinnerComposite;
	}

	/**
	 * Sets the standard {@link GridData} for column <code>1</code> of {@link SpinnerModelDialog}.
	 */
	static GridDataFactory configureColumn_1(Control control) {
		return GridDataFactory.create(control).hintHC(15);
	}

	/**
	 * Sets the standard {@link GridData} for column <code>2</code> of {@link SpinnerModelDialog}.
	 */
	static GridDataFactory configureColumn_2(Control control) {
		return GridDataFactory.create(control).hintHC(20);
	}
}
